<?php
namespace Elementor;

use Elementor\Group_Control_Border;
use Elementor\Repeater;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'Elementor\Jet_Listing_Grid_Widget' ) ) {

	class Jet_Listing_Grid_Widget extends Widget_Base {

		public $is_first     = false;
		public $data         = false;
		public $query_vars   = array();

		public function get_name() {
			return 'jet-listing-grid';
		}

		public function get_title() {
			return __( 'Listing Grid', 'jet-engine' );
		}

		public function get_icon() {
			return 'jet-engine-icon-listing-grid';
		}

		public function get_categories() {
			return array( 'jet-listing-elements' );
		}

		public function get_help_url() {
			return 'https://crocoblock.com/knowledge-base/articles/jetengine-listing-functionality-how-to-create-a-new-listing-to-apply-for-the-certain-post-type/?utm_source=jetengine&utm_medium=listing-grid&utm_campaign=need-help';
		}

		public function register_general_settings() {

			$this->start_controls_section(
				'section_general',
				array(
					'label' => __( 'General', 'jet-engine' ),
				)
			);

			$this->add_control(
				'lisitng_id',
				array(
					'label'   => __( 'Listing', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => '',
					'options' => jet_engine()->listings->get_listings_for_options(),
				)
			);

			$this->add_responsive_control(
				'columns',
				array(
					'label'   => __( 'Columns Number', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 3,
					'options' => array(
						1  => 1,
						2  => 2,
						3  => 3,
						4  => 4,
						5  => 5,
						6  => 6,
						7  => 7,
						8  => 8,
						9  => 9,
						10 => 10,
					),
				)
			);

			$this->add_control(
				'is_archive_template',
				array(
					'label'        => __( 'Use as Archive Template', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'description'  => '',
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'yes',
					'default'      => '',
				)
			);

			$this->add_control(
				'post_status',
				array(
					'label'       => esc_html__( 'Status', 'jet-engine' ),
					'type'        => Controls_Manager::SELECT2,
					'multiple'    => true,
					'label_block' => true,
					'options' => array(
						'publish' => esc_html__( 'Publish', 'jet-engine' ),
						'future'  => esc_html__( 'Future', 'jet-engine' ),
						'draft'   => esc_html__( 'Draft', 'jet-engine' ),
						'pending' => esc_html__( 'Pending Review', 'jet-engine' ),
						'private' => esc_html__( 'Private', 'jet-engine' ),
						'inherit' => esc_html__( 'Inherit', 'jet-engine' ),
					),
					'default'   => array( 'publish' ),
					'condition' => array(
						'is_archive_template!' => 'yes',
					),
				)
			);

			$this->add_control(
				'use_random_posts_num',
				array(
					'label'        => __( 'Use Random posts number', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'yes',
					'default'      => '',
					'condition'    => array(
						'is_archive_template!' => 'yes',
					),
				)
			);

			$this->add_control(
				'random_posts_num_note',
				array(
					'type'            => Controls_Manager::RAW_HTML,
					'raw'             => __( 'Note: the `Posts number` control set min random posts number', 'jet-engine' ),
					'content_classes' => 'elementor-descriptor',
					'condition'       => array(
						'is_archive_template!' => 'yes',
						'use_random_posts_num' => 'yes',
					),
				)
			);

			$this->add_control(
				'posts_num',
				array(
					'label'       => __( 'Posts number', 'jet-engine' ),
					'type'        => Controls_Manager::NUMBER,
					'default'     => 6,
					'min'         => 1,
					'max'         => 1000,
					'step'        => 1,
					'condition'   => array(
						'is_archive_template!' => 'yes',
					),
				)
			);

			$this->add_control(
				'max_posts_num',
				array(
					'label'       => __( 'Max Random Posts number', 'jet-engine' ),
					'type'        => Controls_Manager::NUMBER,
					'default'     => 9,
					'min'         => 1,
					'max'         => 1000,
					'step'        => 1,
					'condition'   => array(
						'is_archive_template!' => 'yes',
						'use_random_posts_num' => 'yes',
					),
				)
			);

			$this->add_control(
				'not_found_message',
				array(
					'label'       => __( 'Not found message', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => __( 'No data was found', 'jet-engine' ),
					'label_block' => true,
				)
			);

			$this->add_control(
				'is_masonry',
				array(
					'label'        => __( 'Is masonry grid', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'yes',
					'default'      => '',
				)
			);

			$this->add_control(
				'equal_columns_height',
				array(
					'label'        => __( 'Equal columns height', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'description'  => __( 'Fits only top level sections of grid item', 'jet-engine' ),
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'yes',
					'default'      => '',
					'condition'    => array(
						'is_masonry!' => 'yes',
					),
				)
			);

			$this->add_control(
				'use_load_more',
				array(
					'label'        => __( 'Load more', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'yes',
					'default'      => '',
				)
			);

			$this->add_control(
				'load_more_id',
				array(
					'label'       => __( 'Load more element ID', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'condition'   => array(
						'use_load_more' => 'yes',
					),
				)
			);

			$this->add_control(
				'use_custom_post_types',
				array(
					'label'        => __( 'Use Custom Post Types', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'yes',
					'default'      => '',
				)
			);

			$this->add_control(
				'custom_post_types',
				array(
					'label'       => esc_html__( 'Post Types', 'jet-engine' ),
					'type'        => Controls_Manager::SELECT2,
					'label_block' => true,
					'multiple'    => true,
					'options'     => jet_engine()->listings->get_post_types_for_options(),
					'condition'   => array(
						'use_custom_post_types' => 'yes',
					),
				)
			);

			do_action( 'jet-engine/listing/after-general-settings', $this );

			$this->end_controls_section();

		}

		public function register_query_settings() {

			$this->start_controls_section(
				'section_posts_query',
				array(
					'label' => __( 'Posts Query', 'jet-engine' ),
				)
			);

			$this->add_control(
				'posts_query_notice',
				array(
					'type' => Controls_Manager::RAW_HTML,
					'raw'  => __( 'Set advanced query parameters', 'jet-engine' ),
				)
			);

			$this->add_control(
				'posts_query_ignored_notice',
				array(
					'type'      => Controls_Manager::RAW_HTML,
					'raw'       => __( 'You select <b>Use as Archive Template</b> option, so other query parameters will be ignored', 'jet-engine' ),
					'condition' => array(
						'is_archive_template' => 'yes',
					),
				)
			);

			$posts_query_repeater = new Repeater();

			$posts_query_repeater->add_control(
				'type',
				array(
					'label'   => __( 'Type', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => '',
					'options' => array(
						'posts_params' => __( 'Posts & Author Parameters', 'jet-engine' ),
						'order_offset' => __( 'Order & Offset', 'jet-engine' ),
						'tax_query'    => __( 'Tax Query', 'jet-engine' ),
						'meta_query'   => __( 'Meta Query', 'jet-engine' ),
						'date_query'   => __( 'Date Query', 'jet-engine' ),
					),
				)
			);

			$posts_query_repeater->add_control(
				'date_query_column',
				array(
					'label'   => __( 'Column', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => '',
					'options' => array(
						'post_date'         => __( 'Post date', 'jet-engine' ),
						'post_date_gmt'     => __( 'Post date GMT', 'jet-engine' ),
						'post_modified'     => __( 'Post modified', 'jet-engine' ),
						'post_modified_gmt' => __( 'Post modified GMT', 'jet-engine' ),
					),
					'condition'   => array(
						'type' => 'date_query'
					),
				)
			);

			$posts_query_repeater->add_control(
				'date_query_after',
				array(
					'label'       => __( 'After', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'description' => __( 'Date to retrieve posts after. Accepts strtotime()-compatible string', 'jet-engine' ),
					'label_block' => true,
					'condition'   => array(
						'type' => 'date_query'
					),
				)
			);

			$posts_query_repeater->add_control(
				'date_query_before',
				array(
					'label'       => __( 'Before', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'description' => __( 'Date to retrieve posts before. Accepts strtotime()-compatible string', 'jet-engine' ),
					'label_block' => true,
					'condition'   => array(
						'type' => 'date_query'
					),
				)
			);

			$posts_query_repeater->add_control(
				'posts_in',
				array(
					'label'       => __( 'Include posts by IDs', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'label_block' => true,
					'default'     => '',
					'description' => __( 'Eg. 12, 24, 33', 'jet-engine' ),
					'condition'   => array(
						'type' => 'posts_params'
					),
				)
			);

			$posts_query_repeater->add_control(
				'posts_not_in',
				array(
					'label'       => __( 'Exclude posts by IDs', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'label_block' => true,
					'default'     => '',
					'description' => __( 'Eg. 12, 24, 33. If this is used in the same query as Include posts by IDs, it will be ignored', 'jet-engine' ),
					'condition'   => array(
						'type' => 'posts_params'
					),
				)
			);

			$posts_query_repeater->add_control(
				'posts_parent',
				array(
					'label'       => __( 'Get child of', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'description' => __( 'Eg. 12, 24, 33', 'jet-engine' ),
					'condition'   => array(
						'type' => 'posts_params'
					),
				)
			);

			$posts_query_repeater->add_control(
				'posts_status',
				array(
					'label'   => __( 'Get posts with status', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'publish',
					'options' => array(
						'publish'    => __( 'Publish', 'jet-engine' ),
						'pending'    => __( 'Pending', 'jet-engine' ),
						'draft'      => __( 'Draft', 'jet-engine' ),
						'auto-draft' => __( 'Auto draft', 'jet-engine' ),
						'future'     => __( 'Future', 'jet-engine' ),
						'private'    => __( 'Private', 'jet-engine' ),
						'trash'      => __( 'Trash', 'jet-engine' ),
						'any'        => __( 'Any', 'jet-engine' ),
					),
					'condition'   => array(
						'type' => 'posts_params'
					),
				)
			);

			$posts_query_repeater->add_control(
				'posts_author',
				array(
					'label'   => __( 'Posts by author', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'any',
					'options' => array(
						'any'     => __( 'Any author', 'jet-engine' ),
						'current' => __( 'Current User', 'jet-engine' ),
						'id'      => __( 'Specific Author ID', 'jet-engine' ),
						'queried' => __( 'Queried User', 'jet-engine' ),
					),
					'condition'   => array(
						'type' => 'posts_params'
					),
				)
			);

			$posts_query_repeater->add_control(
				'posts_author_id',
				array(
					'label'       => __( 'Author ID', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'condition'   => array(
						'type'         => 'posts_params',
						'posts_author' => 'id',
					),
				)
			);

			$posts_query_repeater->add_control(
				'offset',
				array(
					'label'     => __( 'Posts offset', 'jet-engine' ),
					'type'      => Controls_Manager::NUMBER,
					'default'   => '0',
					'min'       => 0,
					'max'       => 100,
					'step'      => 1,
					'condition' => array(
						'type' => 'order_offset'
					),
				)
			);

			$posts_query_repeater->add_control(
				'order',
				array(
					'label'   => __( 'Order', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'DESC',
					'options' => array(
						'ASC'  => __( 'ASC', 'jet-engine' ),
						'DESC' => __( 'DESC', 'jet-engine' ),
					),
					'condition'   => array(
						'type' => 'order_offset'
					),
				)
			);

			$posts_query_repeater->add_control(
				'order_by',
				array(
					'label'   => __( 'Order by', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'date',
					'options' => array(
						'none'          => __( 'None', 'jet-engine' ),
						'ID'            => __( 'ID', 'jet-engine' ),
						'author'        => __( 'Author', 'jet-engine' ),
						'title'         => __( 'Title', 'jet-engine' ),
						'name'          => __( 'Name', 'jet-engine' ),
						'type'          => __( 'Type', 'jet-engine' ),
						'date'          => __( 'Date', 'jet-engine' ),
						'modified'      => __( 'Modified', 'jet-engine' ),
						'parent'        => __( 'Parent', 'jet-engine' ),
						'rand'          => __( 'Rand', 'jet-engine' ),
						'comment_count' => __( 'Comment count', 'jet-engine' ),
						'relevance'     => __( 'Relevance', 'jet-engine' ),
						'menu_order'    => __( 'Menu order', 'jet-engine' ),
						'meta_value'    => __( 'Meta value', 'jet-engine' ),
						'meta_clause'   => __( 'Meta clause', 'jet-engine' ),
						'post__in'      => __( 'Preserve post ID order given in the "Include posts by IDs" option', 'jet-engine' ),
					),
					'condition'   => array(
						'type' => 'order_offset'
					),
				)
			);

			$posts_query_repeater->add_control(
				'meta_key',
				array(
					'label'       => __( 'Meta key to order', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'description' => __( 'Set meta field name to order by', 'jet-engine' ),
					'condition'   => array(
						'type'     => 'order_offset',
						'order_by' => 'meta_value',
					),
				)
			);

			$posts_query_repeater->add_control(
				'meta_clause_key',
				array(
					'label'       => __( 'Meta clause to order', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'description' => __( 'Meta clause name to order by. Clause with this name should be created in Meta Query parameters', 'jet-engine' ),
					'condition'   => array(
						'type'     => 'order_offset',
						'order_by' => 'meta_clause',
					),
				)
			);

			$posts_query_repeater->add_control(
				'meta_type',
				array(
					'label'   => __( 'Meta type', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'CHAR',
					'options' => array(
						'NUMERIC'  => __( 'NUMERIC', 'jet-engine' ),
						'CHAR'     => __( 'CHAR', 'jet-engine' ),
						'DATE'     => __( 'DATE', 'jet-engine' ),
						'DATETIME' => __( 'DATETIME', 'jet-engine' ),
						'DECIMAL'  => __( 'DECIMAL', 'jet-engine' ),
					),
					'condition'   => array(
						'type'     => 'order_offset',
						'order_by' => 'meta_value',
					),
				)
			);

			$posts_query_repeater->add_control(
				'tax_query_taxonomy',
				array(
					'label'   => __( 'Taxonomy', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'options' => jet_engine()->listings->get_taxonomies_for_options(),
					'default' => '',
					'condition' => array(
						'type' => 'tax_query'
					),
				)
			);

			$posts_query_repeater->add_control(
				'tax_query_taxonomy_meta',
				array(
					'label'       => __( 'Taxonomy from meta field', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'description' => __( 'Get taxonomy name from current page meta field', 'jet-engine' ),
					'condition'   => array(
						'type' => 'tax_query'
					),
				)
			);


			$posts_query_repeater->add_control(
				'tax_query_compare',
				array(
					'label'   => __( 'Operator', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'options' => array(
						'IN'         => __( 'IN', 'jet-engine' ),
						'NOT IN'     => __( 'NOT IN', 'jet-engine' ),
						'AND'        => __( 'AND', 'jet-engine' ),
						'EXISTS'     => __( 'EXISTS', 'jet-engine' ),
						'NOT EXISTS' => __( 'NOT EXISTS', 'jet-engine' ),
					),
					'default' => 'IN',
					'condition' => array(
						'type' => 'tax_query'
					),
				)
			);

			$posts_query_repeater->add_control(
				'tax_query_field',
				array(
					'label'   => __( 'Field', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'options' => array(
						'term_id' => __( 'Term ID', 'jet-engine' ),
						'slug'    => __( 'Slug', 'jet-engine' ),
						'name'    => __( 'Name', 'jet-engine' ),
					),
					'default' => 'term_id',
					'condition' => array(
						'type' => 'tax_query'
					),
				)
			);

			$posts_query_repeater->add_control(
				'tax_query_terms',
				array(
					'label'       => __( 'Terms', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'condition'   => array(
						'type' => 'tax_query'
					),
				)
			);

			$posts_query_repeater->add_control(
				'tax_query_terms_meta',
				array(
					'label'       => __( 'Terms from meta field', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'description' => __( 'Get terms IDs from current page meta field', 'jet-engine' ),
					'condition'   => array(
						'type' => 'tax_query'
					),
				)
			);

			do_action( 'jet-engine/listing/after-tax-fields', $posts_query_repeater, $this );

			$posts_query_repeater->add_control(
				'meta_query_key',
				array(
					'label'   => __( 'Key (name/ID)', 'jet-engine' ),
					'type'    => Controls_Manager::TEXT,
					'default' => '',
					'condition' => array(
						'type' => 'meta_query'
					),
				)
			);

			$posts_query_repeater->add_control(
				'meta_query_compare',
				array(
					'label'   => __( 'Operator', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => '=',
					'options' => array(
						'='           => __( 'Equal', 'jet-engine' ),
						'!='          => __( 'Not equal', 'jet-engine' ),
						'>'           => __( 'Greater than', 'jet-engine' ),
						'>='          => __( 'Greater or equal', 'jet-engine' ),
						'<'           => __( 'Less than', 'jet-engine' ),
						'<='          => __( 'Equal or less', 'jet-engine' ),
						'LIKE'        => __( 'Like', 'jet-engine' ),
						'NOT LIKE'    => __( 'Not like', 'jet-engine' ),
						'IN'          => __( 'In', 'jet-engine' ),
						'NOT IN'      => __( 'Not in', 'jet-engine' ),
						'BETWEEN'     => __( 'Between', 'jet-engine' ),
						'NOT BETWEEN' => __( 'Not between', 'jet-engine' ),
						'EXISTS'      => __( 'Exists', 'jet-engine' ),
						'NOT EXISTS'  => __( 'Not Exists', 'jet-engine' ),
					),
					'condition'   => array(
						'type' => 'meta_query',
					),
				)
			);

			$posts_query_repeater->add_control(
				'meta_query_val',
				array(
					'label'       => __( 'Value', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'description' => __( 'For <b>In</b>, <b>Not in</b>, <b>Between</b> and <b>Not between</b> compare separate multiple values with comma', 'jet-engine' ),
					'condition'   => array(
						'type' => 'meta_query',
						'meta_query_compare!' => array( 'EXISTS', 'NOT EXISTS' ),
					),
				)
			);

			$posts_query_repeater->add_control(
				'meta_query_request_val',
				array(
					'label'       => __( 'Or get value from query variable', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'description' => __( 'Set query variable name (from URL or WordPress query var) to get value from', 'jet-engine' ),
					'condition'   => array(
						'type' => 'meta_query',
						'meta_query_compare!' => array( 'EXISTS', 'NOT EXISTS' ),
					),
				)
			);

			$posts_query_repeater->add_control(
				'meta_query_type',
				array(
					'label'   => __( 'Type', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'CHAR',
					'options' => $this->meta_types(),
					'condition'   => array(
						'type' => 'meta_query',
					),
				)
			);

			$posts_query_repeater->add_control(
				'meta_query_clause',
				array(
					'label'       => __( 'Meta Query Clause', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'description' => __( 'Set unique name for current query clause to use it to order posts by this clause', 'jet-engine' ),
					'condition'   => array(
						'type' => 'meta_query',
					),
				)
			);

			do_action( 'jet-engine/listing/after-posts-query-fields', $posts_query_repeater, $this );

			$this->add_control(
				'posts_query',
				array(
					'type'    => Controls_Manager::REPEATER,
					'fields'  => array_values( $posts_query_repeater->get_controls() ),
					'default' => array(),
					'title_field' => '{{{ type }}}',
				)
			);

			$this->add_control(
				'meta_query_relation',
				array(
					'label'   => __( 'Meta query relation', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'AND',
					'options' => array(
						'AND' => __( 'AND', 'jet-engine' ),
						'OR'  => __( 'OR', 'jet-engine' ),
					),
				)
			);

			$this->add_control(
				'tax_query_relation',
				array(
					'label'   => __( 'Tax query relation', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'AND',
					'options' => array(
						'AND' => __( 'AND', 'jet-engine' ),
						'OR'  => __( 'OR', 'jet-engine' ),
					),
				)
			);

			$this->end_controls_section();

		}

		/**
		 * Terms query settings
		 * @return [type] [description]
		 */
		public function register_terms_query_settings() {

			$this->start_controls_section(
				'section_terms_query',
				array(
					'label' => __( 'Terms Query', 'jet-engine' ),
				)
			);

			$this->add_control(
				'terms_query_notice',
				array(
					'type' => Controls_Manager::RAW_HTML,
					'raw'  => __( 'Set advanced query parameters', 'jet-engine' ),

				)
			);

			$this->add_control(
				'terms_query_ignored_notice',
				array(
					'type'      => Controls_Manager::RAW_HTML,
					'raw'       => __( 'You select <b>Use as Archive Template</b> option, so other query parameters will be ignored', 'jet-engine' ),
					'condition' => array(
						'is_archive_template' => 'yes',
					),
				)
			);

			$this->add_control(
				'terms_object_ids',
				array(
					'label'       => __( 'Get terms of posts', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
				)
			);

			$this->add_control(
				'terms_orderby',
				array(
					'label'   => __( 'Order By', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'name',
					'options' => array(
						'name'        => __( 'Name', 'jet-engine' ),
						'slug'        => __( 'Slug', 'jet-engine' ),
						'term_group'  => __( 'Term Group', 'jet-engine' ),
						'term_id'     => __( 'Term ID', 'jet-engine' ),
						'description' => __( 'Description', 'jet-engine' ),
						'parent'      => __( 'Parent', 'jet-engine' ),
						'count'       => __( 'Count', 'jet-engine' ),
						'none'        => __( 'None', 'jet-engine' ),
					),
				)
			);

			$this->add_control(
				'terms_order',
				array(
					'label'   => __( 'Order', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'DESC',
					'options' => array(
						'ASC'  => __( 'ASC', 'jet-engine' ),
						'DESC' => __( 'DESC', 'jet-engine' ),
					),
				)
			);

			$this->add_control(
				'terms_hide_empty',
				array(
					'label'        => __( 'Hide empty', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'description'  => '',
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'true',
					'default'      => 'true',
				)
			);

			$this->add_control(
				'terms_include',
				array(
					'label'       => __( 'Include terms', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'description' => __( 'Comma/space-separated string of term ids to include', 'jet-engine' ),
				)
			);

			$this->add_control(
				'terms_exclude',
				array(
					'label'       => __( 'Exclude terms', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'description' => __( 'Comma/space-separated string of term ids to exclude. Ignore if <b>Include terms</b> not empty', 'jet-engine' ),
				)
			);

			$this->add_control(
				'terms_offset',
				array(
					'label'     => __( 'Offset', 'jet-engine' ),
					'type'      => Controls_Manager::NUMBER,
					'default'   => '0',
					'min'       => 0,
					'max'       => 100,
					'step'      => 1,
				)
			);

			$this->add_control(
				'terms_child_of',
				array(
					'label'       => __( 'Child of', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'description' => __( 'Term ID to retrieve child terms of', 'jet-engine' ),
				)
			);

			$this->add_control(
				'terms_meta_query_heading',
				array(
					'label'     => __( 'Meta Query', 'jet-engine' ),
					'type'      => Controls_Manager::HEADING,
					'separator' => 'before',
				)
			);

			$terms_meta_query = new Repeater();

			$terms_meta_query->add_control(
				'meta_query_key',
				array(
					'label'   => __( 'Key (name/ID)', 'jet-engine' ),
					'type'    => Controls_Manager::TEXT,
					'default' => '',
				)
			);

			$terms_meta_query->add_control(
				'meta_query_compare',
				array(
					'label'   => __( 'Operator', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => '=',
					'options' => array(
						'='           => __( 'Equal', 'jet-engine' ),
						'!='          => __( 'Not equal', 'jet-engine' ),
						'>'           => __( 'Greater than', 'jet-engine' ),
						'>='          => __( 'Greater or equal', 'jet-engine' ),
						'<'           => __( 'Less than', 'jet-engine' ),
						'<='          => __( 'Equal or less', 'jet-engine' ),
						'LIKE'        => __( 'Like', 'jet-engine' ),
						'NOT LIKE'    => __( 'Not like', 'jet-engine' ),
						'IN'          => __( 'In', 'jet-engine' ),
						'NOT IN'      => __( 'Not in', 'jet-engine' ),
						'BETWEEN'     => __( 'Between', 'jet-engine' ),
						'NOT BETWEEN' => __( 'Not between', 'jet-engine' ),
						'EXISTS'      => __( 'Exists', 'jet-engine' ),
						'NOT EXISTS'  => __( 'Not Exists', 'jet-engine' ),
					),
				)
			);

			$terms_meta_query->add_control(
				'meta_query_val',
				array(
					'label'       => __( 'Value', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'description' => __( 'For <b>In</b>, <b>Not in</b>, <b>Between</b> and <b>Not between</b> compare separate multiple values with comma', 'jet-engine' ),
				)
			);

			$terms_meta_query->add_control(
				'meta_query_type',
				array(
					'label'   => __( 'Type', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'CHAR',
					'options' => $this->meta_types(),
				)
			);

			$this->add_control(
				'terms_meta_query',
				array(
					'type'    => Controls_Manager::REPEATER,
					'fields'  => array_values( $terms_meta_query->get_controls() ),
					'default' => array(),
					'title_field' => '{{{ meta_query_key }}}',
				)
			);

			$this->add_control(
				'term_meta_query_relation',
				array(
					'label'   => __( 'Meta query relation', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'AND',
					'options' => array(
						'AND' => __( 'AND', 'jet-engine' ),
						'OR'  => __( 'OR', 'jet-engine' ),
					),
				)
			);

			$this->end_controls_section();

			$this->start_controls_section(
				'section_users_query',
				array(
					'label' => __( 'Users Query', 'jet-engine' ),
				)
			);

			$this->add_control(
				'users_role__in',
				array(
					'label'       => esc_html__( 'Roles', 'jet-engine' ),
					'type'        => Controls_Manager::SELECT2,
					'label_block' => true,
					'multiple'    => true,
					'options'     => $this->get_user_roles(),
					'default'     => array(),
				)
			);

			$this->add_control(
				'users_role__not_in',
				array(
					'label'       => esc_html__( 'Exclude roles', 'jet-engine' ),
					'type'        => Controls_Manager::SELECT2,
					'label_block' => true,
					'multiple'    => true,
					'options'     => $this->get_user_roles(),
					'default'     => array(),
				)
			);

			$this->add_control(
				'users_include',
				array(
					'label'       => __( 'Include users by ID', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'description' => __( 'Comma-separated ID\'s list', 'jet-engine' ),
					'label_block' => true,
				)
			);

			$this->add_control(
				'users_exclude',
				array(
					'label'       => __( 'Exclude users by ID', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'description' => __( 'Comma-separated ID\'s list', 'jet-engine' ),
					'label_block' => true,
				)
			);

			$this->add_control(
				'users_meta_query_heading',
				array(
					'label'     => __( 'Meta Query', 'jet-engine' ),
					'type'      => Controls_Manager::HEADING,
					'separator' => 'before',
				)
			);

			$users_meta_query = new Repeater();

			$users_meta_query->add_control(
				'meta_query_key',
				array(
					'label'   => __( 'Key (name/ID)', 'jet-engine' ),
					'type'    => Controls_Manager::TEXT,
					'default' => '',
				)
			);

			$users_meta_query->add_control(
				'meta_query_compare',
				array(
					'label'   => __( 'Operator', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => '=',
					'options' => array(
						'='           => __( 'Equal', 'jet-engine' ),
						'!='          => __( 'Not equal', 'jet-engine' ),
						'>'           => __( 'Greater than', 'jet-engine' ),
						'>='          => __( 'Greater or equal', 'jet-engine' ),
						'<'           => __( 'Less than', 'jet-engine' ),
						'<='          => __( 'Equal or less', 'jet-engine' ),
						'LIKE'        => __( 'Like', 'jet-engine' ),
						'NOT LIKE'    => __( 'Not like', 'jet-engine' ),
						'IN'          => __( 'In', 'jet-engine' ),
						'NOT IN'      => __( 'Not in', 'jet-engine' ),
						'BETWEEN'     => __( 'Between', 'jet-engine' ),
						'NOT BETWEEN' => __( 'Not between', 'jet-engine' ),
						'EXISTS'      => __( 'Exists', 'jet-engine' ),
						'NOT EXISTS'  => __( 'Not Exists', 'jet-engine' ),
					),
				)
			);

			$users_meta_query->add_control(
				'meta_query_val',
				array(
					'label'       => __( 'Value', 'jet-engine' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => '',
					'label_block' => true,
					'description' => __( 'For <b>In</b>, <b>Not in</b>, <b>Between</b> and <b>Not between</b> compare separate multiple values with comma', 'jet-engine' ),
					'condition'   => array(
						'meta_query_compare!' => array( 'EXISTS', 'NOT EXISTS' ),
					),
				)
			);

			$users_meta_query->add_control(
				'meta_query_type',
				array(
					'label'     => __( 'Type', 'jet-engine' ),
					'type'      => Controls_Manager::SELECT,
					'default'   => 'CHAR',
					'options'   => $this->meta_types(),
					'condition' => array(
						'meta_query_compare!' => array( 'EXISTS', 'NOT EXISTS' ),
					),
				)
			);

			$this->add_control(
				'users_meta_query',
				array(
					'type'    => Controls_Manager::REPEATER,
					'fields'  => array_values( $users_meta_query->get_controls() ),
					'default' => array(),
					'title_field' => '{{{ meta_query_key }}}',
				)
			);

			$this->add_control(
				'users_meta_query_relation',
				array(
					'label'   => __( 'Meta query relation', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'AND',
					'options' => array(
						'AND' => __( 'AND', 'jet-engine' ),
						'OR'  => __( 'OR', 'jet-engine' ),
					),
				)
			);

			$this->end_controls_section();

		}

		public function register_visibility_settings() {

			$this->start_controls_section(
				'section_widget_visibility',
				array(
					'label' => __( 'Widget Visibility', 'jet-engine' ),
				)
			);

			$this->add_control(
				'hide_widget_if',
				array(
					'label'   => __( 'Hide widget if', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => '',
					'options' => jet_engine()->listings->get_widget_hide_options(),
				)
			);

			$this->end_controls_section();

		}

		/**
		 * Register style settings
		 * @return [type] [description]
		 */
		public function register_style_settings() {

			$this->start_controls_section(
				'section_caption_style',
				array(
					'label'      => __( 'Columns', 'jet-engine' ),
					'tab'        => Controls_Manager::TAB_STYLE,
					'show_label' => false,
				)
			);

			$this->add_responsive_control(
				'horizontal_gap',
				array(
					'label' => __( 'Horizontal Gap', 'jet-engine' ),
					'type'  => Controls_Manager::SLIDER,
					'range' => array(
						'px' => array(
							'min' => 0,
							'max' => 100,
						),
					),
					'selectors' => array(
						'{{WRAPPER}} .jet-listing-grid__item' => 'padding-left: calc({{SIZE}}{{UNIT}} / 2); padding-right: calc({{SIZE}}{{UNIT}} / 2);',
						'{{WRAPPER}} .jet-listing-grid__items' => 'margin-left: calc(-{{SIZE}}{{UNIT}} / 2); margin-right: calc(-{{SIZE}}{{UNIT}} / 2); width: calc(100% + {{SIZE}}{{UNIT}});',
					),
				)
			);

			$this->add_responsive_control(
				'vertical_gap',
				array(
					'label' => __( 'Vertical Gap', 'jet-engine' ),
					'type'  => Controls_Manager::SLIDER,
					'range' => array(
						'px' => array(
							'min' => 0,
							'max' => 100,
						),
					),
					'selectors' => array(
						'{{WRAPPER}} .jet-listing-grid__item' => 'padding-top: calc({{SIZE}}{{UNIT}} / 2); padding-bottom: calc({{SIZE}}{{UNIT}} / 2);',
					),
				)
			);

			$this->end_controls_section();

		}

		/**
		 * Register style settings
		 * @return [type] [description]
		 */
		public function register_carousel_settings() {

			$this->start_controls_section(
				'section_carousel',
				array(
					'label' => __( 'Slider', 'jet-engine' ),
				)
			);

			$this->add_control(
				'masonry_notice',
				array(
					'type'      => Controls_Manager::RAW_HTML,
					'raw'       => __( 'Slider settings are disabled for masonry layout', 'jet-engine' ),
					'condition' => array(
						'is_masonry' => 'yes',
					),
				)
			);

			$this->add_control(
				'carousel_enabled',
				array(
					'label'        => __( 'Enable Slider', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'yes',
					'default'      => '',
					'condition'    => array(
						'is_masonry!' => 'yes',
					),
				)
			);

			$this->add_control(
				'slides_to_scroll',
				array(
					'label'     => __( 'Slides to Scroll', 'jet-engine' ),
					'type'      => Controls_Manager::SELECT,
					'default'   => '1',
					'options'   => array(
						1 => 1,
						2 => 2,
						3 => 3,
						4 => 4,
					),
					'condition' => array(
						'columns!' => '1',
						'is_masonry!' => 'yes',
					),
				)
			);

			$this->add_control(
				'arrows',
				array(
					'label'        => __( 'Show Arrows Navigation', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'true',
					'default'      => 'true',
					'condition'    => array(
						'is_masonry!' => 'yes',
					),
				)
			);

			$this->add_control(
				'arrow_icon',
				array(
					'label'   => __( 'Arrow Icon', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'fa fa-angle-left',
					'options' => array(
						'fa fa-angle-left'          => __( 'Angle', 'jet-engine' ),
						'fa fa-chevron-left'        => __( 'Chevron', 'jet-engine' ),
						'fa fa-angle-double-left'   => __( 'Angle Double', 'jet-engine' ),
						'fa fa-arrow-left'          => __( 'Arrow', 'jet-engine' ),
						'fa fa-caret-left'          => __( 'Caret', 'jet-engine' ),
						'fa fa-long-arrow-left'     => __( 'Long Arrow', 'jet-engine' ),
						'fa fa-arrow-circle-left'   => __( 'Arrow Circle', 'jet-engine' ),
						'fa fa-chevron-circle-left' => __( 'Chevron Circle', 'jet-engine' ),
						'fa fa-caret-square-o-left' => __( 'Caret Square', 'jet-engine' ),
					),
					'condition' => array(
						'arrows' => 'true',
						'is_masonry!' => 'yes',
					),
				)
			);

			$this->add_control(
				'dots',
				array(
					'label'        => __( 'Show Dots Navigation', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'true',
					'default'      => '',
					'condition'    => array(
						'is_masonry!' => 'yes',
					),
				)
			);

			$this->add_control(
				'autoplay',
				array(
					'label'        => __( 'Autoplay', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'true',
					'default'      => 'true',
					'condition'    => array(
						'is_masonry!' => 'yes',
					),
				)
			);

			$this->add_control(
				'autoplay_speed',
				array(
					'label'     => __( 'Autoplay Speed', 'jet-engine' ),
					'type'      => Controls_Manager::NUMBER,
					'default'   => 5000,
					'condition' => array(
						'autoplay' => 'true',
						'is_masonry!' => 'yes',
					),
				)
			);

			$this->add_control(
				'infinite',
				array(
					'label'        => __( 'Infinite Loop', 'jet-engine' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => __( 'Yes', 'jet-engine' ),
					'label_off'    => __( 'No', 'jet-engine' ),
					'return_value' => 'true',
					'default'      => 'true',
					'condition'    => array(
						'is_masonry!' => 'yes',
					),
				)
			);

			$this->add_control(
				'effect',
				array(
					'label'   => __( 'Effect', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'slide',
					'options' => array(
						'slide' => __( 'Slide', 'jet-engine' ),
						'fade'  => __( 'Fade', 'jet-engine' ),
					),
					'condition' => array(
						'columns' => '1',
						'is_masonry!' => 'yes',
					),
				)
			);

			$this->add_control(
				'speed',
				array(
					'label'     => __( 'Animation Speed', 'jet-engine' ),
					'type'      => Controls_Manager::NUMBER,
					'default'   => 500,
					'condition' => array(
						'is_masonry!' => 'yes',
					),
				)
			);

			$this->end_controls_section();

		}

		/**
		 * Register carousel styles settings
		 *
		 * @return [type] [description]
		 */
		public function register_carousel_style_settings() {

			$this->start_controls_section(
				'section_slider_style',
				array(
					'label'      => __( 'Slider', 'jet-engine' ),
					'tab'        => Controls_Manager::TAB_STYLE,
					'show_label' => false,
				)
			);

			$this->add_responsive_control(
				'arrows_box_size',
				array(
					'label'      => __( 'Slider arrows box size', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px' ),
					'range'      => array(
						'px' => array(
							'min' => 16,
							'max' => 120,
						),
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}}; line-height: {{SIZE}}{{UNIT}}; margin-top: calc( -{{SIZE}}{{UNIT}}/2 );',
					),
				)
			);

			$this->add_responsive_control(
				'arrows_size',
				array(
					'label'      => __( 'Slider arrows size', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px' ),
					'range'      => array(
						'px' => array(
							'min' => 10,
							'max' => 50,
						),
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon' => 'font-size: {{SIZE}}{{UNIT}};',
					),
				)
			);

			$this->start_controls_tabs( 'tabs_arrow_style' );

			$this->start_controls_tab(
				'tab_arrow_normal',
				array(
					'label' => __( 'Normal', 'jet-engine' ),
				)
			);

			$this->add_control(
				'arrow_color',
				array(
					'label'     => __( 'Color', 'jet-engine' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon' => 'color: {{VALUE}}',
					),
				)
			);

			$this->add_control(
				'arrow_bg_color',
				array(
					'label'     => __( 'Background', 'jet-engine' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon' => 'background: {{VALUE}}',
					),
				)
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'tab_arrow_hover',
				array(
					'label' => __( 'Hover', 'jet-engine' ),
				)
			);

			$this->add_control(
				'arrow_color_hover',
				array(
					'label'     => __( 'Color', 'jet-engine' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon:hover' => 'color: {{VALUE}}',
					),
				)
			);

			$this->add_control(
				'arrow_bg_color_hover',
				array(
					'label'     => __( 'Background', 'jet-engine' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon:hover' => 'background: {{VALUE}}',
					),
				)
			);

			$this->end_controls_tab();

			$this->end_controls_tabs();

			$this->add_control(
				'prev_vert_position',
				array(
					'label'   => __( 'Vertical Position by', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'top',
					'options' => array(
						'top'    => __( 'Top', 'jet-engine' ),
						'bottom' => __( 'Bottom', 'jet-engine' ),
					),
				)
			);

			$this->add_responsive_control(
				'prev_top_position',
				array(
					'label'      => __( 'Top Indent', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px', '%', 'em' ),
					'range'      => array(
						'px' => array(
							'min' => -400,
							'max' => 400,
						),
						'%' => array(
							'min' => -100,
							'max' => 100,
						),
						'em' => array(
							'min' => -50,
							'max' => 50,
						),
					),
					'condition' => array(
						'prev_vert_position' => 'top',
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon.prev-arrow' => 'top: {{SIZE}}{{UNIT}}; bottom: auto;',
					),
				)
			);

			$this->add_responsive_control(
				'prev_bottom_position',
				array(
					'label'      => __( 'Bottom Indent', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px', '%', 'em' ),
					'range'      => array(
						'px' => array(
							'min' => -400,
							'max' => 400,
						),
						'%' => array(
							'min' => -100,
							'max' => 100,
						),
						'em' => array(
							'min' => -50,
							'max' => 50,
						),
					),
					'condition' => array(
						'prev_vert_position' => 'bottom',
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon.prev-arrow' => 'bottom: {{SIZE}}{{UNIT}}; top: auto;',
					),
				)
			);

			$this->add_control(
				'prev_hor_position',
				array(
					'label'   => __( 'Horizontal Position by', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'left',
					'options' => array(
						'left'  => __( 'Left', 'jet-engine' ),
						'right' => __( 'Right', 'jet-engine' ),
					),
				)
			);

			$this->add_responsive_control(
				'prev_left_position',
				array(
					'label'      => __( 'Left Indent', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px', '%', 'em' ),
					'range'      => array(
						'px' => array(
							'min' => -400,
							'max' => 400,
						),
						'%' => array(
							'min' => -100,
							'max' => 100,
						),
						'em' => array(
							'min' => -50,
							'max' => 50,
						),
					),
					'condition' => array(
						'prev_hor_position' => 'left',
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon.prev-arrow' => 'left: {{SIZE}}{{UNIT}}; right: auto;',
					),
				)
			);

			$this->add_responsive_control(
				'prev_right_position',
				array(
					'label'      => __( 'Right Indent', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px', '%', 'em' ),
					'range'      => array(
						'px' => array(
							'min' => -400,
							'max' => 400,
						),
						'%' => array(
							'min' => -100,
							'max' => 100,
						),
						'em' => array(
							'min' => -50,
							'max' => 50,
						),
					),
					'condition' => array(
						'prev_hor_position' => 'right',
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon.prev-arrow' => 'right: {{SIZE}}{{UNIT}}; left: auto;',
					),
				)
			);

			$this->add_control(
				'next_arrow_position',
				array(
					'label'     => __( 'Next Arrow Position', 'jet-engine' ),
					'type'      => Controls_Manager::HEADING,
					'separator' => 'before',
				)
			);

			$this->add_control(
				'next_vert_position',
				array(
					'label'   => __( 'Vertical Position by', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'top',
					'options' => array(
						'top'    => __( 'Top', 'jet-engine' ),
						'bottom' => __( 'Bottom', 'jet-engine' ),
					),
				)
			);

			$this->add_responsive_control(
				'next_top_position',
				array(
					'label'      => __( 'Top Indent', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px', '%', 'em' ),
					'range'      => array(
						'px' => array(
							'min' => -400,
							'max' => 400,
						),
						'%' => array(
							'min' => -100,
							'max' => 100,
						),
						'em' => array(
							'min' => -50,
							'max' => 50,
						),
					),
					'condition' => array(
						'next_vert_position' => 'top',
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon.next-arrow' => 'top: {{SIZE}}{{UNIT}}; bottom: auto;',
					),
				)
			);

			$this->add_responsive_control(
				'next_bottom_position',
				array(
					'label'      => __( 'Bottom Indent', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px', '%', 'em' ),
					'range'      => array(
						'px' => array(
							'min' => -400,
							'max' => 400,
						),
						'%' => array(
							'min' => -100,
							'max' => 100,
						),
						'em' => array(
							'min' => -50,
							'max' => 50,
						),
					),
					'condition' => array(
						'next_vert_position' => 'bottom',
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon.next-arrow' => 'bottom: {{SIZE}}{{UNIT}}; top: auto;',
					),
				)
			);

			$this->add_control(
				'next_hor_position',
				array(
					'label'   => __( 'Horizontal Position by', 'jet-engine' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'right',
					'options' => array(
						'left'  => __( 'Left', 'jet-engine' ),
						'right' => __( 'Right', 'jet-engine' ),
					),
				)
			);

			$this->add_responsive_control(
				'next_left_position',
				array(
					'label'      => __( 'Left Indent', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px', '%', 'em' ),
					'range'      => array(
						'px' => array(
							'min' => -400,
							'max' => 400,
						),
						'%' => array(
							'min' => -100,
							'max' => 100,
						),
						'em' => array(
							'min' => -50,
							'max' => 50,
						),
					),
					'condition' => array(
						'next_hor_position' => 'left',
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon.next-arrow' => 'left: {{SIZE}}{{UNIT}}; right: auto;',
					),
				)
			);

			$this->add_responsive_control(
				'next_right_position',
				array(
					'label'      => __( 'Right Indent', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px', '%', 'em' ),
					'range'      => array(
						'px' => array(
							'min' => -400,
							'max' => 400,
						),
						'%' => array(
							'min' => -100,
							'max' => 100,
						),
						'em' => array(
							'min' => -50,
							'max' => 50,
						),
					),
					'condition' => array(
						'next_hor_position' => 'right',
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider-icon.next-arrow' => 'right: {{SIZE}}{{UNIT}}; left: auto;',
					),
				)
			);

			$this->add_control(
				'dots_styles',
				array(
					'label'     => __( 'Dots Styles', 'jet-engine' ),
					'type'      => Controls_Manager::HEADING,
					'separator' => 'before',
				)
			);

			$this->add_responsive_control(
				'dots_size',
				array(
					'label'      => __( 'Dots Size', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px' ),
					'range'      => array(
						'px' => array(
							'min' => 6,
							'max' => 100,
						),
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider .jet-slick-dots li' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
					),
				)
			);

			$this->add_responsive_control(
				'dots_gap',
				array(
					'label'      => __( 'Dots Gap', 'jet-engine' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px' ),
					'range'      => array(
						'px' => array(
							'min' => 0,
							'max' => 100,
						),
					),
					'selectors'  => array(
						'{{WRAPPER}} .jet-listing-grid__slider .jet-slick-dots li' => 'margin-left: calc( {{SIZE}}{{UNIT}}/2 ); margin-right: calc( {{SIZE}}{{UNIT}}/2 );',
					),
				)
			);

			$this->start_controls_tabs( 'tabs_dots_style' );

			$this->start_controls_tab(
				'tab_dots_normal',
				array(
					'label' => __( 'Normal', 'jet-engine' ),
				)
			);

			$this->add_control(
				'dots_bg_color',
				array(
					'label'     => __( 'Color', 'jet-engine' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .jet-listing-grid__slider .jet-slick-dots li' => 'background: {{VALUE}}',
					),
				)
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'tab_dots_hover',
				array(
					'label' => __( 'Hover', 'jet-engine' ),
				)
			);

			$this->add_control(
				'dots_bg_color_hover',
				array(
					'label'     => __( 'Color', 'jet-engine' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .jet-listing-grid__slider .jet-slick-dots li:hover' => 'background: {{VALUE}}',
					),
				)
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'tab_dots_active',
				array(
					'label' => __( 'Active', 'jet-engine' ),
				)
			);

			$this->add_control(
				'dots_bg_color_active',
				array(
					'label'     => __( 'Color', 'jet-engine' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .jet-listing-grid__slider .jet-slick-dots li.slick-active' => 'background: {{VALUE}}',
					),
				)
			);

			$this->end_controls_tab();

			$this->end_controls_tabs();

			$this->end_controls_section();

		}

		protected function _register_controls() {

			$this->register_general_settings();

			$this->register_query_settings();
			$this->register_terms_query_settings();

			do_action( 'jet-engine/listing/custom-query-settings', $this );

			$this->register_visibility_settings();
			$this->register_carousel_settings();
			$this->register_style_settings();
			$this->register_carousel_style_settings();

		}

		/**
		 * Return meta types list for options
		 * @return [type] [description]
		 */
		public function meta_types() {

			return array(
				'NUMERIC'  => __( 'NUMERIC', 'jet-engine' ),
				'BINARY'   => __( 'BINARY', 'jet-engine' ),
				'CHAR'     => __( 'CHAR', 'jet-engine' ),
				'DATE'     => __( 'DATE', 'jet-engine' ),
				'DATETIME' => __( 'DATETIME', 'jet-engine' ),
				'DECIMAL'  => __( 'DECIMAL', 'jet-engine' ),
				'SIGNED'   => __( 'SIGNED', 'jet-engine' ),
				'UNSIGNED' => __( 'UNSIGNED', 'jet-engine' ),
			);

		}

		/**
		 * Returns all registered user roles
		 *
		 * @return [type] [description]
		 */
		public function get_user_roles() {
			return \Jet_Engine_Tools::get_user_roles();
		}

		/**
		 * Returns widget settings or custom settings
		 *
		 * @return array
		 */
		public function get_widget_settings() {

			$custom_settings = apply_filters( 'jet-engine/listing/grid/custom-settings', false, $this );

			if ( ! empty( $custom_settings ) ) {
				return $custom_settings;
			} else {
				return $this->get_settings();
			}

		}

		/**
		 * Render grid posts
		 *
		 * @return void
		 */
		public function render_posts() {
			jet_engine()->listings->render_listing( $this->get_widget_settings() );
		}

		protected function render() {
			$this->render_posts();
		}

	}

}
