<?php
/**
 * The Video Parser
 *
 * @since      2.0.0
 * @package    RankMath
 * @subpackage RankMath\Schema\Video
 * @author     Rank Math <support@rankmath.com>
 */

namespace RankMathPro\Schema\Video;

use RankMath\Helper;
use RankMath\Schema\DB;
use MyThemeShop\Helpers\Str;

defined( 'ABSPATH' ) || exit;

/**
 * Parser class.
 */
class Parser {

	/**
	 * Post.
	 *
	 * @var WP_Post
	 */
	private $post;

	/**
	 * The Constructor.
	 *
	 * @param  WP_Post $post Post to parse.
	 */
	public function __construct( $post ) {
		$this->post = $post;
	}

	/**
	 * Save video object.
	 */
	public function save() {
		if ( ! $this->can_save() ) {
			return;
		}

		$content = trim( $this->post->post_content . ' ' . $this->get_custom_fields_data() );
		if ( empty( $content ) ) {
			return;
		}

		$content       = apply_filters( 'the_content', $content );
		$allowed_types = apply_filters( 'media_embedded_in_content_allowed_types', [ 'video', 'embed', 'iframe' ] );
		$tags          = implode( '|', $allowed_types );
		$video_data    = [];
		preg_match_all( '#<(?P<tag>' . $tags . ')[^<]*?(?:>[\s\S]*?<\/(?P=tag)>|\s*\/>)#', $content, $matches );
		if ( ! empty( $matches ) && ! empty( $matches[0] ) ) {
			foreach ( $matches[0] as $html ) {
				$video_data = $this->get_metadata( $html );
				if ( ! empty( $video_data ) ) {
					break;
				}
			}
		}

		$video_data = ! empty( $video_data ) ? $video_data : $this->get_links_from_shortcode( $content );
		if ( empty( $video_data ) ) {
			return;
		}

		$schemas   = $this->get_default_schema_data();
		$schemas[] = [
			'@type'            => 'VideoObject',
			'metadata'         => [
				'title'                   => 'Video',
				'type'                    => 'template',
				'shortcode'               => uniqid( 's-' ),
				'isPrimary'               => empty( $schemas ),
				'reviewLocationShortcode' => '[rank_math_rich_snippet]',
				'category'                => '%categories%',
				'tags'                    => '%tags%',
				'isAutoGenerated'         => true,
			],
			'name'             => '%seo_title%',
			'description'      => '%seo_description%',
			'uploadDate'       => ! empty( $video_data['uploadDate'] ) ? $video_data['uploadDate'] : '%date(Y-m-dTH:i:sP)%',
			'thumbnailUrl'     => ! empty( $video_data['thumbnail'] ) ? $video_data['thumbnail'] : '%post_thumbnail%',
			'embedUrl'         => ! empty( $video_data['embed'] ) ? $video_data['src'] : '',
			'contentUrl'       => empty( $video_data['embed'] ) ? $video_data['src'] : '',
			'duration'         => ! empty( $video_data['duration'] ) ? $video_data['duration'] : '',
			'width'            => ! empty( $video_data['width'] ) ? $video_data['width'] : '',
			'height'           => ! empty( $video_data['height'] ) ? $video_data['height'] : '',
			'isFamilyFriendly' => ! empty( $video_data['isFamilyFriendly'] ) ? (bool) $video_data['isFamilyFriendly'] : true,
		];

		foreach ( array_filter( $schemas ) as $schema ) {
			update_post_meta( $this->post->ID, "rank_math_schema_{$schema['@type']}", $schema );
		}
	}

	/**
	 * Get default schema data.
	 */
	private function get_default_schema_data() {
		if (
			! empty( DB::get_schemas( $this->post->ID ) ) ||
			'off' === get_post_meta( $this->post->ID, 'rank_math_rich_snippet', true )
		) {
			return [];
		}

		$default_type = ucfirst( Helper::get_default_schema_type( $this->post->post_type ) );
		if ( ! in_array( $default_type, [ 'Article', 'NewsArticle', 'BlogPosting' ], true ) ) {
			return [];
		}

		return [
			[
				'@type'         => $default_type,
				'metadata'      => [
					'title'     => 'Article',
					'type'      => 'template',
					'isPrimary' => true,
				],
				'headline'      => Helper::get_settings( "titles.pt_{$this->post->post_type}_default_snippet_name" ),
				'description'   => Helper::get_settings( "titles.pt_{$this->post->post_type}_default_snippet_desc" ),
				'datePublished' => '%date(Y-m-dTH:i:sP)%',
				'dateModified'  => '%modified(Y-m-dTH:i:sP)%',
				'image'         => [
					'@type' => 'ImageObject',
					'url'   => '%post_thumbnail%',
				],
				'author'        => [
					'@type' => 'Person',
					'name'  => '%name%',
				],
			],
		];
	}

	/**
	 * Get Video source from the content.
	 *
	 * @param array $html Video Links.
	 *
	 * @return array
	 */
	private function get_metadata( $html ) {
		preg_match_all( '@src="([^"]+)"@', $html, $matches );
		if ( empty( $matches ) || empty( $matches[1] ) ) {
			return false;
		}

		return $this->get_video_metadata( $matches[1][0] );
	}

	/**
	 * Validate Video source.
	 *
	 * @param  string $url Video Source.
	 * @return array
	 */
	private function get_video_metadata( $url ) {
		$networks = [
			'Video\Youtube',
			'Video\Vimeo',
			'Video\WordPress',
		];

		$data = false;
		foreach ( $networks as $network ) {
			$data = \call_user_func( [ '\\RankMathPro\\Schema\\' . $network, 'match' ], $url );
			if ( is_array( $data ) ) {
				break;
			}
		}

		// Save image locally.
		if ( ! empty( $data['thumbnail'] ) ) {
			$data['thumbnail'] = $this->save_video_thumbnail( $data['thumbnail'] );
		}

		return $data;
	}

	/**
	 * Get Video Links from YouTube Embed plugin.
	 *
	 * @param  string $content Post Content.
	 * @return array
	 */
	private function get_links_from_shortcode( $content ) {
		preg_match_all( '%(?:youtube(?:-nocookie)?\.com/(?:[^/]+/.+/|(?:v|e(?:mbed)?)/|.*[?&]v=)|youtu\.be/)([^"&?/ ]{11})%i', $content, $matches );
		if ( empty( $matches ) || empty( $matches[1] ) ) {
			return false;
		}

		foreach ( $matches[1] as $video_id ) {
			$data = $this->get_video_metadata( "https://www.youtube.com/embed/{$video_id}" );
			if ( $data ) {
				break;
			}
		}

		return $data;
	}

	/**
	 * Validate Video source.
	 *
	 * @param  string $url Thumbnail URL.
	 * @return array
	 *
	 * Credits to m1r0 @ https://gist.github.com/m1r0/f22d5237ee93bcccb0d9
	 */
	private function save_video_thumbnail( $url ) {
		if ( ! Helper::get_settings( "titles.pt_{$this->post->post_type}_autogenerate_image" ) ) {
			return false;
		}

		if ( ! class_exists( 'WP_Http' ) ) {
			include_once( ABSPATH . WPINC . '/class-http.php' );
		}

		$http     = new \WP_Http();
		$response = $http->request( $url );
		if ( 200 !== $response['response']['code'] ) {
			return false;
		}

		$upload = wp_upload_bits( basename( $url ), null, $response['body'] );
		if ( ! empty( $upload['error'] ) ) {
			return false;
		}

		$file_path        = $upload['file'];
		$file_name        = basename( $file_path );
		$file_type        = wp_check_filetype( $file_name, null );
		$attachment_title = sanitize_file_name( pathinfo( $file_name, PATHINFO_FILENAME ) );
		$wp_upload_dir    = wp_upload_dir();

		$post_info = [
			'guid'           => $wp_upload_dir['url'] . '/' . $file_name,
			'post_mime_type' => $file_type['type'],
			'post_title'     => $attachment_title,
			'post_content'   => '',
			'post_status'    => 'inherit',
		];

		$attach_id = wp_insert_attachment( $post_info, $file_path, $this->post->ID );

		// Include image.php.
		require_once( ABSPATH . 'wp-admin/includes/image.php' );

		// Define attachment metadata.
		$attach_data = wp_generate_attachment_metadata( $attach_id, $file_path );

		// Assign metadata to attachment.
		wp_update_attachment_metadata( $attach_id, $attach_data );

		return wp_get_attachment_url( $attach_id );
	}

	/**
	 * Can save video object.
	 *
	 * @return bool
	 */
	private function can_save() {
		if (
			wp_is_post_revision( $this->post->ID ) ||
			in_array( $this->post->post_status, [ 'auto-draft', 'trash' ], true ) ||
			! Helper::get_settings( "titles.pt_{$this->post->post_type}_autodetect_video", 'on' )
		) {
			return false;
		}

		$video_schema = get_post_meta( $this->post->ID, 'rank_math_schema_VideoObject', true );
		if ( ! empty( $video_schema ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Get Custom fields data.
	 */
	private function get_custom_fields_data() {
		$custom_fields = Str::to_arr_no_empty( Helper::get_settings( 'sitemap.video_sitemap_custom_fields' ) );
		if ( empty( $custom_fields ) ) {
			return;
		}

		$content = '';
		foreach ( $custom_fields as $custom_field ) {
			$content = $content . ' ' . get_post_meta( $this->post->ID, $custom_field, true );
		}

		return trim( $content );
	}
}
